/*
	This is a part of the source code for Pro/DESKTOP.
	Copyright (C) 1998-1999 Parametric Technology Corporation.
	All rights reserved.
*/

//	RectangularDuplicate.cpp

#include <stdafx.h>

#include "CUtility.h"
#include "CApplication.h"
#include "CEdit.h"

HRESULT RectangularDuplicate(double xNumber, double yNumber, double xSpacing,
														 double ySpacing, ISet *objectSet1)
{

	/*
		Creates an empty aObjectSet and copies the contents of the objectSet
		that is passed to this function.

		Calls the DuplicateLines helper function iteratively to create
		the duplicate objects. After creating the duplicate objects they are 
		auto constrained and the selection is set to the objectSet
	*/

	START_METHOD("RectangularDuplicate")

	HRESULT status = CONV_SUCCESS;

	IGraphicDocument *activeGraphicDoc = NULL;
	IPartDocument *activePart = NULL;
	IWorkplane *activeWorkplane = NULL;
	ISketch *activeSketch = NULL;

	status = GetActivePartWorkplaneSketch(&activePart, &activeGraphicDoc, &activeWorkplane, &activeSketch);
	CHECK_RETURN_STATUS(status)

	IDirection *localX = NULL;
	status = activeWorkplane->GetLocalX(&localX);
	CHECK_RETURN_STATUS(status)

	IDirection *localY = NULL;
	status = activeWorkplane->GetLocalY(&localY);
	CHECK_RETURN_STATUS(status)

	// Create an empty set of lines
	ISet *aLineSet1 = NULL;
	status = (GetCLASS(ObjectSet))->CreateAObjectSet(&aLineSet1);
	CHECK_RETURN_STATUS(status)

	long numberOfObjects = 0;
	status = objectSet1->GetCount(&numberOfObjects);
	CHECK_RETURN_STATUS(status)

	IIt *pIt;
	status = (GetCLASS(It))->CreateAObjectIt(objectSet1,&pIt);
	CHECK_RETURN_STATUS(status)

	ISet *objectSet2 = NULL;
	IDispatch *pDisp = NULL;

	pIt->start(&pDisp); 
	while (pDisp) {
		IGeometric *geometric1 = CAST(IGeometric, pDisp);
		status = aLineSet1->AddMember(geometric1);
		CHECK_RETURN_STATUS(status)
		pDisp = NULL;
		pIt->Next(&pDisp);
	}

	objectSet2 = aLineSet1;

	for( int i=0; i<xNumber; i++ ) {
		for( int j=0; j<yNumber; j++ ) {
			if( i==0 && j==0 )
				continue;

			IVector *newLocalX = NULL;
			status = localX->Multiply(i * xSpacing, &newLocalX);
			CHECK_RETURN_STATUS(status)

			IVector *newLocalY = NULL;
			status = localY->Multiply(j * ySpacing, &newLocalY);
			CHECK_RETURN_STATUS(status)

			IVector *trans = NULL;
			status = newLocalX->Add(CAST(IVectorOrDirection, newLocalY), &trans);
			CHECK_RETURN_STATUS(status)

			IMatrix *transMatrix = NULL;
			status = (GetCLASS(Matrix))->CreateTranslationMatrix(trans, &transMatrix);
			CHECK_RETURN_STATUS(status)

			status = DuplicateLines(activeSketch, objectSet1, transMatrix, objectSet2);
			CHECK_RETURN_STATUS(status)
		}
	}

	// Apply AutoConstrain on the set of lines
	status = activeWorkplane->AutoConstrain(CAST(IObjectOrSet, objectSet2));
	CHECK_RETURN_STATUS(status)

	// Set the selection to the object set
	status = activeGraphicDoc->SetSelection(CAST(IObjectOrSet, objectSet2));
	CHECK_RETURN_STATUS(status)

	END_METHOD("RectangularDuplicate")
}

HRESULT DuplicateLines(ISketch *sketch1, ISet *objectSet1, IMatrix *matrix1, ISet *objectSet2)
{
	/*
		Helper function for RectangularDuplicate()
		Creates duplicates of all objects in the set and adds the newly
		created objects to the existing set of objects
	*/

	START_METHOD("DuplicateLines")

	HRESULT status = CONV_SUCCESS;

	long numberOfObjects = 0;
	status = objectSet1->GetCount(&numberOfObjects);
	CHECK_RETURN_STATUS(status)

	IIt *pIt = NULL;
	status = (GetCLASS(It))->CreateAObjectIt(objectSet1,&pIt);
	CHECK_RETURN_STATUS(status)

	IDispatch *pDisp = NULL;
	pIt->start(&pDisp);
	while (pDisp) {
		IGeometric *geometric1 = CAST(IGeometric, pDisp);

		IGeometry *geometry1 = NULL;
		status = geometric1->GetGeometry(&geometry1);
		CHECK_RETURN_STATUS(status)

		IGeometry *geometry2 = NULL;
		geometry1->Clone(&geometry2);
		CHECK_RETURN_STATUS(status)

		VARIANT_BOOL bVal = FALSE;
		status = geometry2->Transform(matrix1, &bVal);
		CHECK_RETURN_STATUS(status)

		ILine *aLine1 = NULL;
		status = sketch1->CreateLine(CAST(ICurve, geometry2), &aLine1);
		CHECK_RETURN_STATUS(status)

		status = objectSet2->AddMember(aLine1);
		CHECK_RETURN_STATUS(status)

		pDisp = NULL;
			pIt->Next(&pDisp);
	}

	END_METHOD("DuplicateLines")
}
